/* 
 * Wavelet denoise GIMP plugin
 * 
 * wavelet-denoise.c
 * Copyright 2008 by Marco Rossini
 * 
 * Implements the wavelet denoise code of UFRaw by Udi Fuchs
 * which itself bases on the code by Dave Coffin
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * 
 * Instructions:
 * compile with gimp-tool, eg. 'gimp-tool --install wavelet-denoise.c'
 * 
 * Usage: This plugin only works on grayscale images. For colour images
 * you have to first decompose the image according to your needs, eg. to
 * RGB or YBrCr and denoise the individual channels. Then recompose. For
 * composing and decomposing consult the GIMP manual.
 * 
 * FIXME: replace malloc() and free() by GIMP functions
 */

#include <stdlib.h>
#include <math.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#define SQR(x) ((x) * (x))
#define MAX2(x,y) ((x) > (y) ? (x) : (y))
#define MIN2(x,y) ((x) < (y) ? (x) : (y))
#define CLIP(x,min,max) MAX2((min), MIN2((x), (max)))

#define TIMER_READ (1.0 / 10)
#define TIMER_WRITE (1.0 / 7)
#define TIMER_PROCESS (1.0 - TIMER_READ - TIMER_WRITE)

#define MODE_YCBCR 0
#define MODE_RGB 1

static void query (void);
static void run (const gchar * name, gint nparams, const GimpParam * param,
		 gint * nreturn_vals, GimpParam ** return_vals);
static void wavelet_denoise (float *fimg[3], unsigned int width,
			     unsigned int height, float threshold, float a,
			     float b);
static void denoise (GimpDrawable * drawable, GimpPreview * preview);
static void rgb2ycbcr (float *r, float *g, float *b);
static void ycbcr2rgb (float *y, float *cb, float *cr);
static void set_rgb_mode (GtkWidget * w, gpointer data);
static void set_ycbcr_mode (GtkWidget * w, gpointer data);
static void set_preview_channel (GtkWidget * w, gpointer data);
static gboolean user_interface (GimpDrawable * drawable);

GimpPlugInInfo PLUG_IN_INFO = { NULL, NULL, query, run };

MAIN ()
     static void query (void)
{
  static GimpParamDef args[] = {
    {GIMP_PDB_INT32, "run-mode", "Run mode"},
    {GIMP_PDB_IMAGE, "image", "Input image"},
    {GIMP_PDB_DRAWABLE, "drawable", "Input drawable"}
  };

  gimp_install_procedure ("plug-in-wavelet-denoise",
			  "Removes noise in the image using wavelets.",
			  "Removes noise in the image using wavelets.",
			  "Marco Rossini",
			  "Copyright Marco Rossini",
			  "2008",
			  "_Wavelet denoise",
			  "RGB*, GRAY*",
			  GIMP_PLUGIN, G_N_ELEMENTS (args), 0, args, NULL);

  gimp_plugin_menu_register ("plug-in-wavelet-denoise",
			     "<Image>/Filters/Enhance");
}

typedef struct
{
  guint gray_thresholds[2];
  guint colour_thresholds[4];
  guint colour_mode;
  gint preview_channel;
  gboolean preview;
  float times[3];
} wavelet_settings;

static wavelet_settings settings = {
  {1000, 0},			/* gray_thresholds */
  {1000, 1000, 1000, 0},	/* colour_thresholds */
  MODE_YCBCR,			/* colour_mode */
  -1,				/* preview_channel */
  TRUE,				/* preview */
  {2.03, 2.67, 7.47}		/* times */
};

static char *names_ycbcr[] = { "Y", "Cb", "Cr", "Alpha" };
static char *names_rgb[] = { "R", "G", "B", "Alpha" };
static char *names_gray[] = { "Gray", "Alpha" };

static float *fimg[4];
static float *buffer[3];
static gint channels;

GTimer *timer;

static void
run (const gchar * name, gint nparams, const GimpParam * param,
     gint * nreturn_vals, GimpParam ** return_vals)
{
  static GimpParam values[1];
  GimpRunMode run_mode;
  GimpDrawable *drawable;
  gint i;

  timer = g_timer_new ();

  /* Setting mandatory output values */
  *nreturn_vals = 1;
  *return_vals = values;
  values[0].type = GIMP_PDB_STATUS;
  values[0].data.d_status = GIMP_PDB_SUCCESS;

  /* restore settings saved in GIMP core */
  gimp_get_data ("plug-in-wavelet-denoise", &settings);

  drawable = gimp_drawable_get (param[2].data.d_drawable);
  channels = gimp_drawable_bpp (drawable->drawable_id);

  /* allocate buffers */
  /* FIXME: replace by GIMP funcitons */
  for (i = 0; i < channels; i++)
    {
      fimg[i] = (float *) malloc (drawable->width * drawable->height
				  * sizeof (float));
    }
  buffer[1] = (float *) malloc (drawable->width * drawable->height
				* sizeof (float));
  buffer[2] = (float *) malloc (drawable->width * drawable->height
				* sizeof (float));

  /* run GUI if in interactiv mode */
  run_mode = param[0].data.d_int32;
  if (run_mode == GIMP_RUN_INTERACTIVE)
    {
      if (!user_interface (drawable))
	{
	  gimp_drawable_detach (drawable);
	  /* FIXME: should return error status here */
	  return;
	}
    }

  denoise (drawable, NULL);

  /* free buffers */
  /* FIXME: replace by GIMP functions */
  for (i = 0; i < channels; i++)
    {
      free (fimg[i]);
    }
  free (buffer[1]);
  free (buffer[2]);

  gimp_displays_flush ();
  gimp_drawable_detach (drawable);

  /* save settings in the GIMP core */
  gimp_set_data ("plug-in-wavelet-denoise", &settings,
		 sizeof (wavelet_settings));
}

static void
denoise (GimpDrawable * drawable, GimpPreview * preview)
{
  GimpPixelRgn rgn_in, rgn_out;
  gint i, x1, y1, x2, y2, width, height, x, c;
  guchar *line;
  float val[4], times[3], totaltime;

  if (preview)
    {
      gimp_preview_get_position (preview, &x1, &y1);
      gimp_preview_get_size (preview, &width, &height);
      x2 = x1 + width;
      y2 = y1 + height;
    }
  else
    {
      gimp_drawable_mask_bounds (drawable->drawable_id, &x1, &y1, &x2, &y2);
      width = x2 - x1;
      height = y2 - y1;
    }

  gimp_pixel_rgn_init (&rgn_in, drawable, x1, y1, width, height, FALSE,
		       FALSE);
  gimp_pixel_rgn_init (&rgn_out, drawable, x1, y1, width, height,
		       preview == NULL, TRUE);

  /* cache some tiles to make reading/writing faster */
  gimp_tile_cache_ntiles (drawable->width / gimp_tile_width () + 1);

  totaltime = settings.times[0];
  for (i = 0; i < channels; i++)
    {
      if (channels > 2 && settings.colour_thresholds[i] > 0)
	totaltime += settings.times[1];
      else if (channels < 0 && settings.gray_thresholds[i] > 0)
	totaltime += settings.times[1];
    }
  totaltime += settings.times[2];

  /* FIXME: replace by GIMP functions */
  line = (guchar *) malloc (channels * width * sizeof (guchar));

  /* read the full image from GIMP */
  if (!preview)
    gimp_progress_init ("Wavelet denoising...");
  times[0] = g_timer_elapsed (timer, NULL);
  for (i = 0; i < y2 - y1; i++)
    {
      if (!preview && i % 10 == 0)
	gimp_progress_update (settings.times[0] * i
			      / (double) height / totaltime);
      gimp_pixel_rgn_get_row (&rgn_in, line, x1, i + y1, width);

      /* convert pixel values to float and do colour model conv. */
      for (x = 0; x < width; x++)
	{
	  for (c = 0; c < channels; c++)
	    val[c] = (float) line[x * channels + c];
	  if (channels > 2 && settings.colour_mode == MODE_YCBCR)
	    rgb2ycbcr (&(val[0]), &(val[1]), &(val[2]));
	  /* save pixel values and scale for denoising */
	  for (c = 0; c < channels; c++)
	    fimg[c][i * width + x] = sqrt (val[c] * (1 << 24));
	}
    }
  times[0] = g_timer_elapsed (timer, NULL) - times[0];

  /* denoise the channels individually */
  times[1] = g_timer_elapsed (timer, NULL);
  x = 0;			/* variable abuse: counter for number of denoised channels */
  for (c = 0; c < channels; c++)
    {
      double a, b;
      /* in preview mode only process the displayed channel */
      if (preview && settings.preview_channel >= 0 &&
	  settings.preview_channel != c)
	continue;
      buffer[0] = fimg[c];
      b = preview ? 0.0 : settings.times[1] / totaltime;
      a = settings.times[0] + x * settings.times[1];
      a /= totaltime;
      if (channels > 2 && settings.colour_thresholds[c] > 0)
	{
	  wavelet_denoise (buffer, width, height, (float)
			   settings.colour_thresholds[c], a, b);
	  x++;
	}
      if (channels < 3 && settings.gray_thresholds > 0)
	{
	  wavelet_denoise (buffer, width, height, (float)
			   settings.gray_thresholds[c], a, b);
	  x++;
	}
    }
  times[1] = g_timer_elapsed (timer, NULL) - times[1];
  times[1] /= x;

  /* retransform the image data
     (in preview mode calculate only the selected channel) */
  for (c = 0; c < channels; c++)
    {
      if (preview && settings.preview_channel >= 0 &&
	  settings.preview_channel != c)
	continue;
      for (i = 0; i < width * height; i++)
	fimg[c][i] = SQR (fimg[c][i]) / 16777216.0;
    }

  /* in single channel preview mode set all channels equal */
  /* FIXME: put this in previous loop */
  if (preview && settings.preview_channel >= 0)
    {
      for (c = 0; c < channels; c++)
	{
	  if (settings.preview_channel == c)
	    continue;
	  for (i = 0; i < width * height; i++)
	    fimg[c][i] = fimg[settings.preview_channel][i];
	}
    }

  /* set alpha to full opacity in single channel preview mode */
  /* FIXME: put this in previous loop */
  if (preview && settings.preview_channel >= 0 && channels % 2 == 0)
    for (i = 0; i < width * height; i++)
      fimg[channels - 1][i] = 255;

  /* convert to RGB if necessary */
  if (channels > 2 && settings.colour_mode == MODE_YCBCR &&
      (!preview || settings.preview_channel == -1))
    {
      for (i = 0; i < width * height; i++)
	ycbcr2rgb (&(fimg[0][i]), &(fimg[1][i]), &(fimg[2][i]));
    }

  /* write the image back to GIMP */
  times[2] = g_timer_elapsed (timer, NULL);
  for (i = 0; i < height; i++)
    {
      if (!preview)
	gimp_progress_update ((settings.times[0] + channels
			       * settings.times[1] + settings.times[2]
			       * i / (double) height) / totaltime);

      /* scale and convert back to guchar */
      for (x = 0; x < width; x++)
	{
	  for (c = 0; c < channels; c++)
	    {
	      /* avoid rounding errors !!! */
	      line[x * channels + c] =
		(guchar) CLIP (floor (fimg[c][i * width + x] + 0.5), 0, 255);
	    }
	}
      gimp_pixel_rgn_set_row (&rgn_out, line, x1, i + y1, width);
    }
  times[2] = g_timer_elapsed (timer, NULL) - times[2];

  if (!preview)
    {
      settings.times[0] = times[0];
      settings.times[1] = times[1];
      settings.times[2] = times[2];
    }

  /* FIXME: replace by gimp functions */
  free (line);

  if (preview)
    {
      gimp_drawable_preview_draw_region (GIMP_DRAWABLE_PREVIEW (preview),
					 &rgn_out);
      return;
    }
  gimp_drawable_flush (drawable);
  gimp_drawable_merge_shadow (drawable->drawable_id, TRUE);
  gimp_drawable_update (drawable->drawable_id, x1, y1, width, height);
}

/* code copied from UFRaw which (originates from dcraw) */
static void
hat_transform (float *temp, float *base, int st, int size, int sc)
{
  int i;
  for (i = 0; i < sc; i++)
    temp[i] = 2 * base[st * i] + base[st * (sc - i)] + base[st * (i + sc)];
  for (; i + sc < size; i++)
    temp[i] = 2 * base[st * i] + base[st * (i - sc)] + base[st * (i + sc)];
  for (; i < size; i++)
    temp[i] = 2 * base[st * i] + base[st * (i - sc)]
      + base[st * (2 * size - 2 - (i + sc))];
}

/* actual denoising algorithm */
/* code copied from UFRaw (originates from dcraw) */
static void
wavelet_denoise (float *fimg[3], unsigned int width,
		 unsigned int height, float threshold, float a, float b)
{
  float *temp, thold;
  unsigned int i, lev, lpass, hpass, size, col, row;
  float noise[] = {
    0.8002, 0.2735, 0.1202, 0.0585, 0.0291, 0.0152, 0.0080, 0.0044
  };

  size = width * height;

  /* FIXME: replace by GIMP functions */
  temp = (float *) malloc (MAX2 (width, height) * sizeof (float));

  hpass = 0;
  for (lev = 0; lev < 5; lev++)
    {
      if (b != 0)
	gimp_progress_update (a + b * lev / 5.0);
      lpass = ((lev & 1) + 1);
      for (row = 0; row < height; row++)
	{
	  hat_transform (temp, fimg[hpass] + row * width, 1, width, 1 << lev);
	  for (col = 0; col < width; col++)
	    {
	      fimg[lpass][row * width + col] = temp[col] * 0.25;
	    }
	}
      if (b != 0)
	gimp_progress_update (a + b * (lev + 0.4) / 5.0);
      for (col = 0; col < width; col++)
	{
	  hat_transform (temp, fimg[lpass] + col, width, height, 1 << lev);
	  for (row = 0; row < height; row++)
	    {
	      fimg[lpass][row * width + col] = temp[row] * 0.25;
	    }
	}
      if (b != 0)
	gimp_progress_update (a + b * (lev + 0.8) / 5.0);
      thold = threshold * noise[lev];
      for (i = 0; i < size; i++)
	{
	  fimg[hpass][i] -= fimg[lpass][i];
	  if (fimg[hpass][i] < -thold)
	    fimg[hpass][i] += thold;
	  else if (fimg[hpass][i] > thold)
	    fimg[hpass][i] -= thold;
	  else
	    fimg[hpass][i] = 0;
	  if (hpass)
	    fimg[0][i] += fimg[hpass][i];
	}
      hpass = lpass;
    }

  for (i = 0; i < size; i++)
    fimg[0][i] = fimg[0][i] + fimg[lpass][i];

  /* FIXME: replace by GIMP functions */
  free (temp);
}

static gboolean
user_interface (GimpDrawable * drawable)
{
  /* colour mode frame */
  GtkWidget *fr_mode, *mode_radio[2], *mode_vbox;
  GSList *mode_list;

  /* channel select frame */
  GtkWidget *fr_channel, *channel_radio[5], *channel_vbox;
  GSList *channel_list;

  /* threshold frame */
  GtkWidget *fr_threshold, *thr_label[4], *thr_spin[4];
  GtkWidget *thr_hbox[4], *thr_vbox, *thr_scale[4];
  GtkObject *thr_adj[4];

  /* dialog */
  GtkWidget *dialog, *dialog_hbox, *dialog_vbox;
  GtkWidget *preview;
  gboolean run;
  gint i;
  char **names;

  GtkWidget **radios_labels[] = { channel_radio, thr_label };

  if (channels < 3)
    {
      names = names_gray;
    }
  else
    {
      if (settings.colour_mode == MODE_YCBCR)
	names = names_ycbcr;
      else
	names = names_rgb;
    }

  gimp_ui_init ("Wavelet denoise", FALSE);

  /* prepare the preview */
  preview = gimp_drawable_preview_new (drawable, &settings.preview);
  gtk_box_set_homogeneous (GTK_BOX (preview), FALSE);
  g_signal_connect_swapped (preview, "invalidated", G_CALLBACK (denoise),
			    drawable);
  gtk_widget_show (preview);

  /* prepare the colour mode frame */
  if (channels >= 3)
    {
      fr_mode = gtk_frame_new ("Color model");
      mode_vbox = gtk_vbox_new (FALSE, 0);
      gtk_container_border_width (GTK_CONTAINER (mode_vbox), 5);
      mode_radio[0] = gtk_radio_button_new_with_label (NULL,
						       channels ==
						       3 ? "YCbCr" :
						       "YCbCr(A)");
      mode_list =
	gtk_radio_button_get_group (GTK_RADIO_BUTTON (mode_radio[0]));
      mode_radio[1] =
	gtk_radio_button_new_with_label (mode_list,
					 channels == 3 ? "RGB" : "RGB(A)");
      if (settings.colour_mode == MODE_YCBCR)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (mode_radio[0]), 1);
      else
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (mode_radio[1]), 1);
      g_signal_connect (mode_radio[0], "toggled",
			G_CALLBACK (set_ycbcr_mode), radios_labels);
      g_signal_connect (mode_radio[1], "toggled",
			G_CALLBACK (set_rgb_mode), radios_labels);
      g_signal_connect_swapped (mode_radio[0], "toggled",
				G_CALLBACK (gimp_preview_invalidate),
				preview);
      g_signal_connect_swapped (mode_radio[1], "toggled",
				G_CALLBACK (gimp_preview_invalidate),
				preview);
      gtk_container_add (GTK_CONTAINER (fr_mode), mode_vbox);
      gtk_box_pack_start (GTK_BOX (mode_vbox), mode_radio[0], FALSE,
			  FALSE, 0);
      gtk_box_pack_start (GTK_BOX (mode_vbox), mode_radio[1], FALSE,
			  FALSE, 0);
      gtk_widget_show (mode_radio[0]);
      gtk_widget_show (mode_radio[1]);
      gtk_widget_show (mode_vbox);
      gtk_widget_show (fr_mode);
    }
  else
    {
      fr_mode = NULL;
    }

  /* prepare the channel select frame */
  if (channels > 1)
    {
      fr_channel = gtk_frame_new ("Preview channel");
      channel_vbox = gtk_vbox_new (FALSE, 0);
      gtk_container_border_width (GTK_CONTAINER (channel_vbox), 5);
      gtk_container_add (GTK_CONTAINER (fr_channel), channel_vbox);

      channel_list = NULL;
      for (i = 0; i < channels; i++)
	{
	  channel_radio[i] =
	    gtk_radio_button_new_with_label (channel_list, names[i]);
	  if (settings.preview_channel == i)
	    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
					  (channel_radio[i]), 1);
	  g_signal_connect (channel_radio[i], "toggled",
			    G_CALLBACK (set_preview_channel), (gpointer) i);
	  g_signal_connect_swapped (channel_radio[i], "toggled",
				    G_CALLBACK (gimp_preview_invalidate),
				    preview);
	  gtk_box_pack_start (GTK_BOX (channel_vbox), channel_radio[i], FALSE,
			      FALSE, 0);
	  gtk_widget_show (channel_radio[i]);
	  channel_list =
	    gtk_radio_button_get_group (GTK_RADIO_BUTTON (channel_radio[i]));
	}
      channel_radio[4] =
	gtk_radio_button_new_with_label (channel_list, "All");
      if (settings.preview_channel == -1)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (channel_radio[4]),
				      1);
      g_signal_connect (channel_radio[4], "toggled",
			G_CALLBACK (set_preview_channel), (gpointer) - 1);
      g_signal_connect_swapped (channel_radio[4], "toggled",
				G_CALLBACK (gimp_preview_invalidate),
				preview);
      gtk_box_pack_start (GTK_BOX (channel_vbox), channel_radio[4], FALSE,
			  FALSE, 0);
      gtk_widget_show (channel_radio[4]);
      gtk_widget_show (channel_vbox);
      gtk_widget_show (fr_channel);
    }
  else
    {
      fr_channel = NULL;
    }

  /* prepare the threshold frame */
  fr_threshold = gtk_frame_new ("Channel thresholds");
  gtk_widget_show (fr_threshold);
  thr_vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (fr_threshold), thr_vbox);
  gtk_container_border_width (GTK_CONTAINER (thr_vbox), 5);
  gtk_widget_show (thr_vbox);

  for (i = 0; i < channels; i++)
    {
      thr_hbox[i] = gtk_hbox_new (FALSE, 10);
      thr_label[i] = gtk_label_new (names[i]);
      if (channels == 2 || channels == 4)
	gtk_label_set_width_chars (GTK_LABEL (thr_label[i]), 5);
      else if (channels == 1)
	gtk_label_set_width_chars (GTK_LABEL (thr_label[i]), 4);
      else
	gtk_label_set_width_chars (GTK_LABEL (thr_label[i]), 2);
      gtk_misc_set_alignment (GTK_MISC (thr_label[i]), 0.0, 0.0);
      if (channels > 2)
	thr_adj[i] =
	  gtk_adjustment_new (settings.colour_thresholds[i], 0, 10000, 100,
			      100, 0);
      else
	thr_adj[i] =
	  gtk_adjustment_new (settings.gray_thresholds[i], 0, 10000, 100, 100,
			      0);
      thr_spin[i] = gtk_spin_button_new (GTK_ADJUSTMENT (thr_adj[i]), 1, 0);
      thr_scale[i] = gtk_hscale_new (GTK_ADJUSTMENT (thr_adj[i]));
      gtk_scale_set_draw_value (GTK_SCALE (thr_scale[i]), FALSE);
      gtk_box_pack_start (GTK_BOX (thr_vbox), thr_hbox[i], FALSE, FALSE, 0);
      gtk_box_pack_start (GTK_BOX (thr_hbox[i]), thr_label[i], FALSE,
			  FALSE, 0);
      gtk_box_pack_start (GTK_BOX (thr_hbox[i]), thr_scale[i], TRUE, TRUE, 0);
      gtk_box_pack_start (GTK_BOX (thr_hbox[i]), thr_spin[i], FALSE,
			  FALSE, 0);
      g_signal_connect_swapped (thr_adj[i], "value_changed",
				G_CALLBACK (gimp_preview_invalidate),
				preview);
      if (channels > 2)
	g_signal_connect (thr_adj[i], "value_changed",
			  G_CALLBACK (gimp_int_adjustment_update),
			  &(settings.colour_thresholds[i]));
      else
	g_signal_connect (thr_adj[i], "value_changed",
			  G_CALLBACK (gimp_int_adjustment_update),
			  &(settings.gray_thresholds[i]));
      gtk_widget_show (thr_scale[i]);
      gtk_widget_show (thr_spin[i]);
      gtk_widget_show (thr_label[i]);
      gtk_widget_show (thr_hbox[i]);
    }

  /* prepeare the dialog */
  dialog_vbox = gtk_vbox_new (FALSE, 0);
  dialog_hbox = gtk_hbox_new (FALSE, 10);
  gtk_box_set_homogeneous (GTK_BOX (dialog_hbox), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog_vbox), 10);
  dialog = gimp_dialog_new ("Wavelet denoise", "hallo", NULL, 0,
			    gimp_standard_help_func,
			    "plug-in-wavelet-denoise", GTK_STOCK_CANCEL,
			    GTK_RESPONSE_CANCEL, GTK_STOCK_OK,
			    GTK_RESPONSE_OK, NULL);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), dialog_vbox);
  gtk_box_pack_start (GTK_BOX (dialog_vbox), preview, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (dialog_vbox), dialog_hbox, FALSE, FALSE, 0);
  if (fr_channel)
    gtk_box_pack_start (GTK_BOX (dialog_hbox), fr_channel, TRUE, TRUE, 0);
  if (fr_mode)
    gtk_box_pack_start (GTK_BOX (dialog_hbox), fr_mode, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (dialog_vbox), fr_threshold, FALSE, FALSE, 0);
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
  gtk_widget_show (dialog_hbox);
  gtk_widget_show (dialog_vbox);
  gtk_widget_show (dialog);

  run = (gimp_dialog_run (GIMP_DIALOG (dialog)) == GTK_RESPONSE_OK);

  /* FIXME: destroy all widgets */
  gtk_widget_destroy (dialog);

  return run;
}

static void
set_rgb_mode (GtkWidget * w, gpointer data)
{
  /* change labels in the dialog */
  GtkWidget **radios = ((GtkWidget ***) data)[0];
  GtkWidget **labels = ((GtkWidget ***) data)[1];
  gtk_button_set_label (GTK_BUTTON (radios[0]), "R");
  gtk_button_set_label (GTK_BUTTON (radios[1]), "G");
  gtk_button_set_label (GTK_BUTTON (radios[2]), "B");
  gtk_label_set_text (GTK_LABEL (labels[0]), "R");
  gtk_label_set_text (GTK_LABEL (labels[1]), "G");
  gtk_label_set_text (GTK_LABEL (labels[2]), "B");
  settings.colour_mode = MODE_RGB;
}

static void
set_ycbcr_mode (GtkWidget * w, gpointer data)
{
  /* change labels in the dialog */
  GtkWidget **radios = ((GtkWidget ***) data)[0];
  GtkWidget **labels = ((GtkWidget ***) data)[1];
  gtk_button_set_label (GTK_BUTTON (radios[0]), "Y");
  gtk_button_set_label (GTK_BUTTON (radios[1]), "Cb");
  gtk_button_set_label (GTK_BUTTON (radios[2]), "Cr");
  gtk_label_set_text (GTK_LABEL (labels[0]), "Y");
  gtk_label_set_text (GTK_LABEL (labels[1]), "Cb");
  gtk_label_set_text (GTK_LABEL (labels[2]), "Cr");
  settings.colour_mode = MODE_YCBCR;
}

static void
set_preview_channel (GtkWidget * w, gpointer data)
{
  if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w)))
    return;
  settings.preview_channel = (gint) data;
}

static void
rgb2ycbcr (float *r, float *g, float *b)
{
  /* using JPEG conversion here - expecting all channels to be
   * in [0:255] range */
  static float y, cb, cr;
  y = 0.2990 * *r + 0.5870 * *g + 0.1140 * *b;
  cb = -0.1687 * *r - 0.3313 * *g + 0.5000 * *b + 128.0;
  cr = 0.5000 * *r - 0.4187 * *g - 0.0813 * *b + 128.0;
  *r = y;
  *g = cb;
  *b = cr;
}

static void
ycbcr2rgb (float *y, float *cb, float *cr)
{
  /* using JPEG conversion here - expecting all channels to be
   * in [0:255] range */
  static float r, g, b;
  r = *y + 1.40200 * (*cr - 128.0);
  g = *y - 0.34414 * (*cb - 128.0) - 0.71414 * (*cr - 128.0);
  b = *y + 1.77200 * (*cb - 128.0);
  *y = r;
  *cb = g;
  *cr = b;
}
